# How to use this script:
# Run as python preprocess.py <path_to_folder> <export_path>
# Python version >= 3.5

# <path_to_folder> is TopSpin Experiment path
# <export_path> is were projections will be exported to
# which contains different expno folders
# Read Settings section for further explanation

import numpy as np
import pathlib as pl
import scipy.interpolate as sint
import scipy.fftpack as fft
import sys

#---------Settings for preprocessing--------------------------

# All the sliding window lengths that should be calculated
sw_lengths = [128, 1024]

# The expno folder numbers for the first experiment/projection angle
expno_first = 10101

# The expno folder numbers for the last experiment/projection angle
# This expno is included (inclusive range)
expno_last = 10101 + 899

# Zero-filling y/n?
zero_fill = False

#---------------End of Settings-------------------------------


import_path = pl.Path(sys.argv[1])
export_path = pl.Path(sys.argv[2])


def powerSpectrum(spectrum):
    return spectrum * spectrum.conj()


def doSlidingWindow(noise_block, window_size, increment):
    size = noise_block.shape[0]
    accum = None

    if zero_fill:
        accum = np.zeros(2 * window_size, dtype=np.float64)
    else:
        accum = np.zeros(window_size, dtype=np.float64)

    for i in range(0, size - window_size + 1, increment):
        window = noise_block[i:i+window_size]
        if zero_fill:
            window = np.concatenate([window, np.zeros(window_size)])
        spectrum = fft.fft(window)
        accum = accum + powerSpectrum(spectrum)
    return accum


def readAndProcess(fid_path, window_size):
    increment = int(window_size / 7)

    if not fid_path.exists():
        print('This file does not exist: ' + str(fid_path))
        sys.exit()

    with fid_path.open() as fid_file:
        fid = np.fromfile(fid_file, dtype=np.int32)

    real = fid[::2]
    imag = fid[1::2]
    imag = imag * 1j
    noise_block = real + imag

    return doSlidingWindow(noise_block, window_size, increment)


def helper(path_and_ws):
    return readAndProcess(path_and_ws[0], path_and_ws[1])


def normalize(projection):
    projection_length = projection.shape[0]

    baseline_region_start = int(0.1 * projection_length)
    baseline_region_end = int(0.2 * projection_length)
    baseline_avg = np.average(np.concatenate(
        [projection[baseline_region_start: baseline_region_end], 
            projection[-baseline_region_end: -baseline_region_start]]))

    projection_norm = projection - baseline_avg
    projection_norm[:baseline_region_start] = 0
    projection_norm[-baseline_region_start:] = 0

    area = np.sum(projection_norm) * (2048 / projection_length)
    projection_norm = projection_norm * (1000 / area)

    return projection_norm


def correctFreqRef(projection):
    proj_center = int(projection.shape[0] / 2)
    projection[proj_center:] = projection[-1:proj_center-1:-1]
    projection[:proj_center] = projection[proj_center-1::-1]
    return projection


def correctProjection(projection):
    projection = correctFreqRef(projection)
    projection = normalize(projection)

    return projection


def preprocessAll(fid_paths, window_lengths, export_path):

    for window_length in window_lengths:
        fid_paths_and_ws = map(lambda path: (path, window_length), fid_paths)
        projections = list(map(helper, fid_paths_and_ws))
        stacked_projections = np.stack(projections, axis=0)
        corrected_projections = np.apply_along_axis(
            correctProjection, arr=stacked_projections, axis=1)
        corrected_projections.tofile(
            str(export_path / ('proj_' + str(window_length) + '.raw')))


def main():
    all_expnos = range(expno_first, expno_last + 1)
    fid_paths = list(map(lambda expno: import_path / str(expno) / 'ser', all_expnos))
    export = pl.Path(export_path)

    preprocessAll(fid_paths, sw_lengths, export)


if __name__ == '__main__':
    main()