% Supporting information for the manuscript
%
% Multidimensional encoding of restricted and anisotropic diffusion by double rotation of the q-vector
% Hong Jiang, Leo Svenningsson, Daniel Topgaard
% Submitted to Magn. Reson.
%
% Matlab code for generating figure 2.
% Equation numbers refer to main text.

clear all

% Input parameters
pars_g1d.np = 1000; % Number of points in waveform
pars_g1d.epsilon_up = 0.015; % Quarter-sine ramp up duration (total duration tau = 1)
pars_g1d.epsilon_plateau = 0; % Plateau
pars_g1d.epsilon_down = 0.06; % Half-sine ramp down duration

pars_dor.Delta_psi2 = 2*pi; % Total angle of rotation about axis 2
pars_dor.n = 4; % DOR frequency ratio

b_delta = 0.5; % b-tensor anisotropy, defined in Eq. (34)
b_eta = 0.25; % b-tensor asymmetry, defined in Eq. (35)


% Calculations
% Generate 1D waveform g1d
[g1d,t] = dor_g1d(pars_g1d);

Delta_psi2 = pars_dor.Delta_psi2; % Total angle of rotation about axis 2
n = pars_dor.n; % DOR frequency ratio

% Axis inclinations zeta1 and zeta2
if n == 0 % DORn with n = 0 is same as q-MAS Eriksson, J. Magn. Reson. 226, 13 (2013). doi: 10.1016/j.jmr.2012.10.015
    zeta1 = 0; 
    zeta2 = acos(1/sqrt(3));
else
    zeta1 = pi/2;
    zeta2 = -acos(1/sqrt(3));
end

q = cumsum(g1d); % Eq. (49). Note: dt and gamma in Eqs. (24) and (26) cancel in Eq. (25) and can be omitted altogether.
q = mean([q flipud(q)],2); % Removes asymmetry introduced by cumsum
b = sum(q.^2); % Eq. (51)
psi2 = Delta_psi2/b*cumsum(q.^2); % Eq. (50)
psi2 = mean([psi2 Delta_psi2-flipud(psi2)],2); % Removes asymmetry introduced by cumsum

grot   = Delta_psi2.*q.^3/b; % Eq. (53)

psi1     =       n*psi2; % Eq. (54)
psiplus  = (n + 1)*psi2;
psiminus = (n - 1)*psi2;

a0     = cos(zeta1)*cos(zeta2); % Eq. (55)
a1     = sin(zeta1)*sin(zeta2);
a2     = cos(zeta1)*sin(zeta2);
aplus  = sin(zeta1)*(cos(zeta2)+1)/2;
aminus = sin(zeta1)*(cos(zeta2)-1)/2;
   
gdor.x =  g1d.*(        aplus.*cos(psiplus) +        aminus.*cos(psiminus) + a2.*cos(psi2)) ... % Eq. (52)
       + grot.*(-(n+1).*aplus.*sin(psiplus) - (n-1).*aminus.*sin(psiminus) - a2.*sin(psi2));
gdor.y =  g1d.*(        aplus.*sin(psiplus) -        aminus.*sin(psiminus) + a2.*sin(psi2)) ...
       + grot.*( (n+1).*aplus.*cos(psiplus) - (n-1).*aminus.*cos(psiminus) + a2.*cos(psi2));
gdor.z =  g1d.*( a0 - a1.*cos(psi1)) ...
       + grot.*(    n*a1.*sin(psi1));

gdor = dor_g1d_zetas2gwf(pars_g1d,pars_dor);

gpas.x = gdor.x*sqrt(1-b_delta*(1+b_eta)); % Eq. (56)
gpas.y = gdor.y*sqrt(1-b_delta*(1-b_eta));
gpas.z = gdor.z*sqrt(1+2*b_delta);

% Sanity check
gx = gpas.x;
gy = gpas.y;
gz = gpas.z;

qx = cumsum(gx); % Eq. (4)
qy = cumsum(gy);
qz = cumsum(gz);

bxx = sum(qx.*qx); % Eq. (6) in Basser, J. Magn. Reson. B 103, 247 (1994). doi: 10.1006/jmrb.1994.1037
byy = sum(qy.*qy);
bzz = sum(qz.*qz);
bxy = sum(qx.*qy);
bxz = sum(qx.*qz);
byz = sum(qy.*qz);

[[bxx bxy bxz]; [bxy byy byz]; [bxz byz bzz]]/(bxx + byy + bzz)
% [b bxx + byy + bzz]

% Plotting
tbaseline = [-.02 1.02];

lw = 4*1;
fs = 4*6;

left = .08;
width = (1-left - .03)/5;
bottom = .07;
height = .2;
xlim = [-.1 1.1]; ylim = 1.3*[ -1.1 1.1];
xtick = [0 .5 1]; ytick = [-1 0 1];
ticklength = .04*[1 1];

figure(1), clf
axh1 = axes('position',[left+0*width .65-height width height]);
hg1d = plot(t,g1d);
set([hg1d],'LineWidth',lw,'Color',0*[1 1 1],'LineStyle','-')

axh2 = axes('position',[left+2*width .99-height width height]);
hq = plot(t,q/max(q));
set([hq],'LineWidth',lw,'Color',0*[1 1 1],'LineStyle','-')

axh3 = axes('position',[left+4*width .65-height width height]);
hpsiplus = plot(t,psiplus/2/pi);
hold on
hpsi1 = plot(t,psi1/2/pi);
hpsiminus = plot(t,psiminus/2/pi);
hpsi2 = plot(t,psi2/2/pi);
set([hpsi2],'LineWidth',lw,'Color',0*[1 1 1],'LineStyle','-')
set([hpsiplus],'LineWidth',lw,'Color',.75*[1 1 1],'LineStyle','-')
set([hpsi1],'LineWidth',lw,'Color',.5*[1 1 1],'LineStyle','-')
set([hpsiminus],'LineWidth',lw,'Color',.25*[1 1 1],'LineStyle','-')

axh4 = axes('position',[left+2*width  bottom width height]);
hgdorx = plot(t,gdor.x);
hold on
hgdory = plot(t,gdor.y);
hgdorz = plot(t,gdor.z);

axh5 = axes('position',[left+4*width  bottom width height]);
hgx = plot(t,gx);
hold on
hgy = plot(t,gy);
hgz = plot(t,gz);

axh6 = axes('position',[left+2*width  .65-height width height]);
hgrot = plot(t,grot);
set([hgrot],'LineWidth',lw,'Color',0*[1 1 1],'LineStyle','-')

set([hgdorx; hgx],'LineWidth',lw,'Color',[1 0 0],'LineStyle','-')
set([hgdory; hgy],'LineWidth',lw,'Color',[0 .7 0],'LineStyle','-')
set([hgdorz; hgz],'LineWidth',lw,'Color',[.3 .3 1],'LineStyle','-')

set([axh1; axh2; axh3; axh4; axh5; axh6],'XLim',xlim,'YLim',ylim,'Box','off','XTick',xtick,'YTick',ytick,'TickDir','out','TickLength',ticklength,...
    'LineWidth',lw,'FontSize',fs)
set(axh3, 'YLim',5*[ -1.1 1.1],'YTick',[-5 0 5])

set([axh1; axh2; axh4; axh5; axh6],'YTickLabel',{'';'0';''})

xlabel(axh1,'\itt\rm / \tau')
ylabel(axh1,'\itg\rm_{1D}(\itt\rm)')
ylabel(axh2,'\itq\rm(\itt\rm)')
ylabel(axh3,'\psi(\itt\rm) / 2\pi')
ylabel(axh4,'\bfg\rm_{DOR}(\itt\rm)')
ylabel(axh5,'\bfg\rm(\itt\rm)')
ylabel(axh6,'\itg\rm_{rot}(\itt\rm)')

set(gcf, 'PaperUnits','centimeters','PaperPosition', 4*[0 0 8.31 5.99],'PaperSize', 4*[8.31 5.99]); 
eval(['print dor_waveform_recipe.pdf -dpdf -loose'])
